<?php
/*
Plugin Name: BuddyBoss Complete Analytics - Improved
Description: Comprehensive Analytics Dashboard with All Requested Metrics - Enhanced Version
Version: 4.0
Author: mobinteg (improved by Manus AI)
*/

if (!defined('ABSPATH')) {
    exit;
}

class BuddyBoss_Complete_Analytics_Improved {

    private $cache_duration = 3600; // 1 hour cache
    private $date_range = '3 months'; // Default date range

    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_assets'));
        add_action('wp_ajax_bb_analytics_get_data', array($this, 'ajax_get_data'));
        add_action('wp_ajax_bb_analytics_update_range', array($this, 'ajax_update_range'));
        
        // Initialize plugin tables if needed
        register_activation_hook(__FILE__, array($this, 'create_analytics_tables'));
    }

    public function add_admin_menu() {
        add_menu_page(
            'Complete Analytics',
            'Complete Analytics',
            'manage_options',
            'buddyboss-analytics',
            array($this, 'display_full_dashboard'),
            'dashicons-chart-bar',
            6
        );
    }

    public function enqueue_assets($hook) {
        if ($hook === 'toplevel_page_buddyboss-analytics') {
            // CSS
            wp_enqueue_style('bb-analytics-css', plugins_url('assets/css/analytics.css', __FILE__));
            
            // JS
            wp_enqueue_script('chart-js', 'https://cdn.jsdelivr.net/npm/chart.js');
            wp_enqueue_script('bb-analytics-js', plugins_url('assets/js/analytics.js', __FILE__), array('jquery', 'chart-js'), null, true);
            
            // Localize data for JS
            $analytics_data = $this->get_all_chart_data();
            wp_localize_script('bb-analytics-js', 'bbAnalyticsData', array(
                'data' => $analytics_data,
                'ajaxurl' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('bb_analytics_nonce')
            ));
        }
    }

    /**
     * Create analytics tables for tracking custom metrics
     */
    public function create_analytics_tables() {
        global $wpdb;
        
        $charset_collate = $wpdb->get_charset_collate();
        
        // User sessions table
        $table_name = $wpdb->prefix . 'bb_user_sessions';
        $sql = "CREATE TABLE $table_name (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            user_id bigint(20) NOT NULL,
            session_start datetime DEFAULT CURRENT_TIMESTAMP NOT NULL,
            session_end datetime DEFAULT NULL,
            duration int(11) DEFAULT 0,
            posts_viewed int(11) DEFAULT 0,
            pages_visited text,
            PRIMARY KEY (id),
            KEY user_id (user_id),
            KEY session_start (session_start)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
        
        // Post views table
        $table_name = $wpdb->prefix . 'bb_post_views';
        $sql = "CREATE TABLE $table_name (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            user_id bigint(20) NOT NULL,
            post_id bigint(20) NOT NULL,
            view_time int(11) DEFAULT 0,
            view_date datetime DEFAULT CURRENT_TIMESTAMP NOT NULL,
            PRIMARY KEY (id),
            KEY user_id (user_id),
            KEY post_id (post_id),
            KEY view_date (view_date)
        ) $charset_collate;";
        
        dbDelta($sql);
        
        // Search log table
        $table_name = $wpdb->prefix . 'bb_search_log';
        $sql = "CREATE TABLE $table_name (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            user_id bigint(20) DEFAULT NULL,
            search_term varchar(255) NOT NULL,
            results_count int(11) DEFAULT 0,
            search_date datetime DEFAULT CURRENT_TIMESTAMP NOT NULL,
            ip_address varchar(45),
            PRIMARY KEY (id),
            KEY user_id (user_id),
            KEY search_term (search_term),
            KEY search_date (search_date)
        ) $charset_collate;";
        
        dbDelta($sql);
    }

    /**
     * AJAX handler for getting analytics data
     */
    public function ajax_get_data() {
        check_ajax_referer('bb_analytics_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        $section = sanitize_text_field($_POST['section']);
        $date_range = sanitize_text_field($_POST['date_range'] ?? '3 months');
        
        $this->date_range = $date_range;
        
        switch ($section) {
            case 'members':
                $data = $this->get_members_data();
                break;
            case 'activity':
                $data = $this->get_activity_data();
                break;
            case 'groups':
                $data = $this->get_groups_data();
                break;
            default:
                $data = array();
        }
        
        wp_send_json_success($data);
    }

    /**
     * Get cached data or execute query
     */
    private function get_cached_data($cache_key, $callback) {
        $cached_data = get_transient($cache_key);
        if ($cached_data !== false) {
            return $cached_data;
        }
        
        $data = call_user_func($callback);
        set_transient($cache_key, $data, $this->cache_duration);
        
        return $data;
    }

    /**
     * Get date range for queries
     */
    private function get_date_range_sql() {
        return date('Y-m-d H:i:s', strtotime('-' . $this->date_range));
    }

    public function display_full_dashboard() {
        echo '<div class="wrap bb-analytics-wrap">';
        echo '<h1>BuddyBoss Community Analytics - Enhanced</h1>';
        
        // Date range selector
        echo '<div class="bb-date-range-selector" style="display:none;">';
        echo '<label for="bb-date-range">Date Range: </label>';
        echo '<select id="bb-date-range" name="date_range">';
        echo '<option value="1 week">Last Week</option>';
        echo '<option value="1 month">Last Month</option>';
        echo '<option value="3 months" selected>Last 3 Months</option>';
        echo '<option value="6 months">Last 6 Months</option>';
        echo '<option value="1 year">Last Year</option>';
        echo '<option value="all">All Time</option>';
        echo '</select>';
        echo '</div>';
        
        // Dashboard summary row
        echo '<div class="bb-dashboard-summary">';
        $this->display_summary_metrics();
        echo '</div>';
        
        // Main content tabs
        echo '<div class="bb-analytics-tabs">';
        echo '<nav class="nav-tab-wrapper">';
        echo '<a href="#members" class="nav-tab nav-tab-active">Members</a>';
        echo '<a href="#activity" class="nav-tab">Activity</a>';
        echo '<a href="#groups" class="nav-tab">Groups</a>';
        echo '<a href="#network" class="nav-tab">Network</a>';
        echo '<a href="#engagement" class="nav-tab">Engagement</a>';
        echo '<a href="#search" class="nav-tab">Search</a>';
        echo '<a href="#messages" class="nav-tab">Messages</a>';
        echo '</nav>';
        
        // Tab content
        echo '<div id="members" class="bb-tab-content active">';
        $this->display_enhanced_members_section();
        echo '</div>';
        
        echo '<div id="activity" class="bb-tab-content">';
        $this->display_activity_section();
        echo '</div>';
        
        echo '<div id="groups" class="bb-tab-content">';
        $this->display_groups_section();
        echo '</div>';
        
        echo '<div id="network" class="bb-tab-content">';
        $this->display_network_section();
        echo '</div>';

        echo '<div id="engagement" class="bb-tab-content">';
        $this->display_engagement_section();
        echo '</div>';

        echo '<div id="search" class="bb-tab-content">';
        $this->display_search_section();
        echo '</div>';

        echo '<div id="messages" class="bb-tab-content">';
        $this->display_messages_section();
        echo '</div>';
        
        echo '</div>'; // Close tabs
        echo '</div>'; // Close wrap
    }

    private function display_summary_metrics() {
        $cache_key = 'bb_analytics_summary_' . md5($this->date_range);
        $metrics = $this->get_cached_data($cache_key, array($this, 'get_summary_metrics'));
        
        echo '<div class="bb-summary-container">';
        
        foreach ($metrics as $metric) {
            echo '<div class="bb-summary-card">';
            echo '<div class="bb-summary-value">' . number_format($metric['value']) . '</div>';
            echo '<div class="bb-summary-label">' . esc_html($metric['label']) . '</div>';
            if (isset($metric['change'])) {
                $change_class = $metric['change'] >= 0 ? 'positive' : 'negative';
                echo '<div class="bb-summary-change ' . $change_class . '">';
                echo ($metric['change'] >= 0 ? '+' : '') . number_format($metric['change'], 1) . '%';
                echo '</div>';
            }
            echo '</div>';
        }
        
        echo '</div>';
    }

    private function get_summary_metrics() {
        global $wpdb;
        
        $date_range = $this->get_date_range_sql();
        
        // Total members
        $total_members = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->users}");
        
        // Active members (using last_activity meta or recent activity)
        $active_members = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(DISTINCT u.ID) FROM {$wpdb->users} u
             LEFT JOIN {$wpdb->usermeta} um ON u.ID = um.user_id AND um.meta_key = 'last_activity'
             LEFT JOIN {$wpdb->prefix}bp_activity ba ON u.ID = ba.user_id
             WHERE (um.meta_value >= %s OR ba.date_recorded >= %s)",
            $date_range, $date_range
        ));
        
        // Total groups
        $total_groups = 0;
        if ($this->table_exists($wpdb->prefix . 'bp_groups')) {
            $total_groups = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}bp_groups");
        }
        
        // Total activity
        $total_activity = 0;
        if ($this->table_exists($wpdb->prefix . 'bp_activity')) {
            $total_activity = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM {$wpdb->prefix}bp_activity WHERE date_recorded >= %s",
                $date_range
            ));
        }
        
        // Calculate changes (compared to previous period)
        $prev_date_range = date('Y-m-d H:i:s', strtotime('-' . $this->date_range . ' -' . $this->date_range));
        
        $prev_active_members = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(DISTINCT u.ID) FROM {$wpdb->users} u
             LEFT JOIN {$wpdb->usermeta} um ON u.ID = um.user_id AND um.meta_key = 'last_activity'
             LEFT JOIN {$wpdb->prefix}bp_activity ba ON u.ID = ba.user_id
             WHERE (um.meta_value BETWEEN %s AND %s OR ba.date_recorded BETWEEN %s AND %s)",
            $prev_date_range, $date_range, $prev_date_range, $date_range
        ));
        
        $active_change = $prev_active_members > 0 ? 
            (($active_members - $prev_active_members) / $prev_active_members) * 100 : 0;
        
        return array(
            array('label' => 'Total Members', 'value' => $total_members),
            array('label' => 'Active Members', 'value' => $active_members, 'change' => $active_change),
            array('label' => 'Total Groups', 'value' => $total_groups),
            array('label' => 'Recent Activity', 'value' => $total_activity)
        );
    }

    private function display_enhanced_members_section() {
        echo '<div class="bb-analytics-section">';
        echo '<h2><i class="dashicons dashicons-groups"></i> Member Analytics</h2>';
        
        // Growth Chart
        echo '<div class="bb-subsection">';
        echo '<h3>Member Growth Over Time</h3>';
        echo '<div class="bb-chart-container"><canvas id="membersGrowthChart"></canvas></div>';
        echo '</div>';
        
        // Member Statistics
        $this->display_member_statistics();
        
        // Member Types
        $this->display_member_types();
        
        // Popular Members
        $this->display_popular_members_improved();
        
        echo '</div>';
    }

    private function display_member_statistics() {
        global $wpdb;
        
        echo '<div class="bb-subsection">';
        echo '<h3>Member Statistics</h3>';
        
        $date_range = $this->get_date_range_sql();
        
        // Get comprehensive member stats
        $stats = array();
        
        // Total members
        $stats['total'] = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->users}");
        
        // Active members
        $stats['active'] = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(DISTINCT u.ID) FROM {$wpdb->users} u
             LEFT JOIN {$wpdb->usermeta} um ON u.ID = um.user_id AND um.meta_key = 'last_activity'
             LEFT JOIN {$wpdb->prefix}bp_activity ba ON u.ID = ba.user_id
             WHERE (um.meta_value >= %s OR ba.date_recorded >= %s)",
            $date_range, $date_range
        ));
        
        // New members in period
        $stats['new'] = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$wpdb->users} WHERE user_registered >= %s",
            $date_range
        ));
        
        // Members with posts
        if ($this->table_exists($wpdb->prefix . 'bp_activity')) {
            $stats['contributors'] = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(DISTINCT user_id) FROM {$wpdb->prefix}bp_activity 
                 WHERE type = 'activity_update' AND date_recorded >= %s",
                $date_range
            ));
        } else {
            $stats['contributors'] = 0;
        }
        
        echo '<div class="bb-stats-grid">';
        
        echo '<div class="bb-stat-card">';
        echo '<div class="bb-stat-value">' . number_format($stats['total']) . '</div>';
        echo '<div class="bb-stat-label">Total Members</div>';
        echo '</div>';
        
        echo '<div class="bb-stat-card">';
        echo '<div class="bb-stat-value">' . number_format($stats['active']) . '</div>';
        echo '<div class="bb-stat-label">Active Members</div>';
        echo '</div>';
        
        echo '<div class="bb-stat-card">';
        echo '<div class="bb-stat-value">' . number_format($stats['new']) . '</div>';
        echo '<div class="bb-stat-label">New Members</div>';
        echo '</div>';
        
        echo '<div class="bb-stat-card">';
        echo '<div class="bb-stat-value">' . number_format($stats['contributors']) . '</div>';
        echo '<div class="bb-stat-label">Contributors</div>';
        echo '</div>';
        
        echo '<div class="bb-stat-card">';
        $engagement_rate = $stats['total'] > 0 ? round(($stats['active'] / $stats['total']) * 100, 1) : 0;
        echo '<div class="bb-stat-value">' . $engagement_rate . '%</div>';
        echo '<div class="bb-stat-label">Engagement Rate</div>';
        echo '</div>';
        
        echo '</div>';
        echo '</div>';
    }

    private function display_member_types() {
        global $wpdb;
        
        echo '<div class="bb-subsection">';
        echo '<h3>Member Types Distribution</h3>';
        
        // Get member types from user meta
        $member_types = $wpdb->get_results("
            SELECT 
                COALESCE(um.meta_value, 'Subscriber') as member_type,
                COUNT(*) as count
            FROM {$wpdb->users} u
            LEFT JOIN {$wpdb->usermeta} um ON u.ID = um.user_id AND um.meta_key = 'member_type'
            GROUP BY COALESCE(um.meta_value, 'Subscriber')
            ORDER BY count DESC
        ");
        
        if (!empty($member_types)) {
            echo '<div class="bb-chart-container"><canvas id="memberTypesChart"></canvas></div>';
            
            echo '<div class="bb-table-responsive">';
            echo '<table class="bb-analytics-table">';
            echo '<thead><tr><th>Type</th><th>Count</th><th>Percentage</th></tr></thead>';
            echo '<tbody>';
            
            $total = array_sum(array_column($member_types, 'count'));
            
            foreach ($member_types as $type) {
                $percentage = $total > 0 ? round(($type->count / $total) * 100, 2) : 0;
                echo '<tr>';
                echo '<td>' . esc_html($type->member_type) . '</td>';
                echo '<td>' . number_format($type->count) . '</td>';
                echo '<td>' . $percentage . '%</td>';
                echo '</tr>';
            }
            echo '</tbody></table>';
            echo '</div>';
        } else {
            echo '<p>No member type data available.</p>';
        }
        
        echo '</div>';
    }

    private function display_popular_members_improved() {
        global $wpdb;
        
        echo '<div class="bb-subsection">';
        echo '<h2><i class="dashicons dashicons-awards"></i> Top Members</h2>';
        
        $date_range = $this->get_date_range_sql();
        
        // Optimized query using JOINs instead of subqueries
        $popular_members_query = "
            SELECT 
                u.ID, 
                u.display_name,
                u.user_email,
                COALESCE(um.meta_value, 'Subscriber') as member_type,
                COALESCE(activity_counts.activity_count, 0) as activity_count,
                COALESCE(follower_counts.follower_count, 0) as follower_count
            FROM {$wpdb->users} u
            LEFT JOIN {$wpdb->usermeta} um ON u.ID = um.user_id AND um.meta_key = 'member_type'
        ";
        
        // Add activity counts if bp_activity table exists
        if ($this->table_exists($wpdb->prefix . 'bp_activity')) {
            $popular_members_query .= "
                LEFT JOIN (
                    SELECT user_id, COUNT(*) as activity_count
                    FROM {$wpdb->prefix}bp_activity
                    WHERE date_recorded >= '{$date_range}'
                    GROUP BY user_id
                ) activity_counts ON u.ID = activity_counts.user_id
            ";
        }
        
        // Add follower counts if bp_follow table exists
        if ($this->table_exists($wpdb->prefix . 'bp_follow')) {
            $popular_members_query .= "
                LEFT JOIN (
                    SELECT leader_id, COUNT(*) as follower_count
                    FROM {$wpdb->prefix}bp_follow
                    GROUP BY leader_id
                ) follower_counts ON u.ID = follower_counts.leader_id
            ";
        }
        
        $popular_members_query .= "
            WHERE (activity_counts.activity_count > 0 OR follower_counts.follower_count > 0)
            ORDER BY (COALESCE(activity_counts.activity_count, 0) + COALESCE(follower_counts.follower_count, 0) * 2) DESC
            LIMIT 25
        ";
        
        $popular_members = $wpdb->get_results($popular_members_query);
        
        if (!empty($popular_members)) {
            echo '<div class="bb-table-responsive">';
            echo '<table class="bb-analytics-table">';
            echo '<thead><tr><th>Member</th><th style="display:none;">Type</th><th>Activities</th><th>Followers</th><th>Score</th></tr></thead>';
            echo '<tbody>';
            
            foreach ($popular_members as $member) {
                $score = ($member->activity_count * 1) + ($member->follower_count * 2);
                echo '<tr>';
                echo '<td><a href="' . get_edit_user_link($member->ID) . '">' . esc_html($member->display_name) . '</a></td>';
                echo '<td style="display:none;">' . esc_html($member->member_type) . '</td>';
                echo '<td>' . number_format($member->activity_count) . '</td>';
                echo '<td>' . number_format($member->follower_count) . '</td>';
                echo '<td>' . number_format($score) . '</td>';
                echo '</tr>';
            }
            echo '</tbody></table>';
            echo '</div>';
        } else {
            echo '<p>No member activity data available for the selected period.</p>';
        }
        
        echo '</div>';
    }

    private function display_activity_section() {
        global $wpdb;
        
        echo '<div class="bb-analytics-section">';
        echo '<h2><i class="dashicons dashicons-admin-post"></i> Activity Analytics</h2>';
        
        if (!$this->table_exists($wpdb->prefix . 'bp_activity')) {
            echo '<div class="bb-notice bb-notice-warning">';
            echo '<p>BuddyPress/BuddyBoss activity component is not active or tables are missing.</p>';
            echo '</div>';
            echo '</div>';
            return;
        }
        
        $date_range = $this->get_date_range_sql();
        
        // Activity over time
        echo '<div class="bb-subsection">';
        echo '<h3>Activity Over Time</h3>';
        echo '<div class="bb-chart-container"><canvas id="activityChart"></canvas></div>';
        echo '</div>';
        
        // Activity types
        echo '<div class="bb-subsection">';
        echo '<h3>Activity Types</h3>';
        
        $activity_types = $wpdb->get_results($wpdb->prepare("
            SELECT type, COUNT(*) as count
            FROM {$wpdb->prefix}bp_activity
            WHERE date_recorded >= %s
            GROUP BY type
            ORDER BY count DESC
        ", $date_range));
        
        if (!empty($activity_types)) {
            echo '<div class="bb-chart-container"><canvas id="activityTypesChart"></canvas></div>';
            
            echo '<div class="bb-table-responsive">';
            echo '<table class="bb-analytics-table">';
            echo '<thead><tr><th>Activity Type</th><th>Count</th><th>Percentage</th></tr></thead>';
            echo '<tbody>';
            
            $total_activities = array_sum(array_column($activity_types, 'count'));
            
            foreach ($activity_types as $type) {
                $percentage = $total_activities > 0 ? round(($type->count / $total_activities) * 100, 2) : 0;
                echo '<tr>';
                echo '<td>' . esc_html($this->format_activity_type($type->type)) . '</td>';
                echo '<td>' . number_format($type->count) . '</td>';
                echo '<td>' . $percentage . '%</td>';
                echo '</tr>';
            }
            echo '</tbody></table>';
            echo '</div>';
        }
        
        echo '</div>';
        
        // Top contributors
        echo '<div class="bb-subsection">';
        echo '<h3>Top Contributors</h3>';
        
        $top_contributors = $wpdb->get_results($wpdb->prepare("
            SELECT 
                u.display_name,
                COUNT(ba.id) as activity_count
            FROM {$wpdb->prefix}bp_activity ba
            JOIN {$wpdb->users} u ON ba.user_id = u.ID
            WHERE ba.date_recorded >= %s
            GROUP BY ba.user_id, u.display_name
            ORDER BY activity_count DESC
            LIMIT 10
        ", $date_range));
        
        if (!empty($top_contributors)) {
            echo '<div class="bb-table-responsive">';
            echo '<table class="bb-analytics-table">';
            echo '<thead><tr><th>User</th><th>Activities</th></tr></thead>';
            echo '<tbody>';
            
            foreach ($top_contributors as $contributor) {
                echo '<tr>';
                echo '<td>' . esc_html($contributor->display_name) . '</td>';
                echo '<td>' . number_format($contributor->activity_count) . '</td>';
                echo '</tr>';
            }
            echo '</tbody></table>';
            echo '</div>';
        }
        
        echo '</div>';
        echo '</div>';
    }

    private function display_groups_section() {
        global $wpdb;
        
        echo '<div class="bb-analytics-section">';
        echo '<h2><i class="dashicons dashicons-groups"></i> Groups Analytics</h2>';
        
        if (!$this->table_exists($wpdb->prefix . 'bp_groups')) {
            echo '<div class="bb-notice bb-notice-warning">';
            echo '<p>BuddyPress/BuddyBoss groups component is not active or tables are missing.</p>';
            echo '</div>';
            echo '</div>';
            return;
        }
        
        $date_range = $this->get_date_range_sql();
        
        // Groups overview
        echo '<div class="bb-subsection">';
        echo '<h3>Groups Overview</h3>';
        
        $total_groups = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}bp_groups");
        $new_groups = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$wpdb->prefix}bp_groups WHERE date_created >= %s",
            $date_range
        ));
        
        echo '<div class="bb-stats-grid">';
        echo '<div class="bb-stat-card">';
        echo '<div class="bb-stat-value">' . number_format($total_groups) . '</div>';
        echo '<div class="bb-stat-label">Total Groups</div>';
        echo '</div>';
        
        echo '<div class="bb-stat-card">';
        echo '<div class="bb-stat-value">' . number_format($new_groups) . '</div>';
        echo '<div class="bb-stat-label">New Groups</div>';
        echo '</div>';
        echo '</div>';
        
        echo '</div>';
        
        // Groups by status
        echo '<div class="bb-subsection">';
        echo '<h3>Groups by Status</h3>';
        
        $groups_by_status = $wpdb->get_results("
            SELECT status, COUNT(*) as count
            FROM {$wpdb->prefix}bp_groups
            GROUP BY status
            ORDER BY count DESC
        ");
        
        if (!empty($groups_by_status)) {
            echo '<div class="bb-table-responsive">';
            echo '<table class="bb-analytics-table">';
            echo '<thead><tr><th>Status</th><th>Count</th><th>Percentage</th></tr></thead>';
            echo '<tbody>';
            
            foreach ($groups_by_status as $status) {
                $percentage = $total_groups > 0 ? round(($status->count / $total_groups) * 100, 2) : 0;
                echo '<tr>';
                echo '<td>' . esc_html(ucfirst($status->status)) . '</td>';
                echo '<td>' . number_format($status->count) . '</td>';
                echo '<td>' . $percentage . '%</td>';
                echo '</tr>';
            }
            echo '</tbody></table>';
            echo '</div>';
        }
        
        echo '</div>';
        
        // Most active groups
        if ($this->table_exists($wpdb->prefix . 'bp_activity')) {
            echo '<div class="bb-subsection">';
            echo '<h3>Most Active Groups</h3>';
            
            $active_groups = $wpdb->get_results($wpdb->prepare("
                SELECT 
                    g.name,
                    COUNT(ba.id) as activity_count
                FROM {$wpdb->prefix}bp_groups g
                LEFT JOIN {$wpdb->prefix}bp_activity ba ON g.id = ba.item_id AND ba.component = 'groups'
                WHERE ba.date_recorded >= %s
                GROUP BY g.id, g.name
                HAVING activity_count > 0
                ORDER BY activity_count DESC
                LIMIT 10
            ", $date_range));
            
            if (!empty($active_groups)) {
                echo '<div class="bb-table-responsive">';
                echo '<table class="bb-analytics-table">';
                echo '<thead><tr><th>Group</th><th>Activities</th></tr></thead>';
                echo '<tbody>';
                
                foreach ($active_groups as $group) {
                    echo '<tr>';
                    echo '<td>' . esc_html($group->name) . '</td>';
                    echo '<td>' . number_format($group->activity_count) . '</td>';
                    echo '</tr>';
                }
                echo '</tbody></table>';
                echo '</div>';
            } else {
                echo '<p>No group activity data available for the selected period.</p>';
            }
            
            echo '</div>';
        }
        
        echo '</div>';
    }

    private function display_network_section() {
        global $wpdb;
        
        echo '<div class="bb-analytics-section">';
        echo '<h2><i class="dashicons dashicons-networking"></i> Network Analytics</h2>';
        
        $date_range = $this->get_date_range_sql();
        
        // Friendships
        if ($this->table_exists($wpdb->prefix . 'bp_friends')) {
            echo '<div class="bb-subsection">';
            echo '<h3>Friendships</h3>';
            
            $total_friendships = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}bp_friends WHERE is_confirmed = 1");
            $new_friendships = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM {$wpdb->prefix}bp_friends WHERE is_confirmed = 1 AND date_created >= %s",
                $date_range
            ));
            
            echo '<div class="bb-stats-grid">';
            echo '<div class="bb-stat-card">';
            echo '<div class="bb-stat-value">' . number_format($total_friendships) . '</div>';
            echo '<div class="bb-stat-label">Total Friendships</div>';
            echo '</div>';
            
            echo '<div class="bb-stat-card">';
            echo '<div class="bb-stat-value">' . number_format($new_friendships) . '</div>';
            echo '<div class="bb-stat-label">New Friendships</div>';
            echo '</div>';
            echo '</div>';
            
            echo '</div>';
        }
        
        // Follows
        if ($this->table_exists($wpdb->prefix . 'bp_follow')) {
            echo '<div class="bb-subsection">';
            echo '<h3>Follows</h3>';
            
            $total_follows = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}bp_follow");
            $new_follows = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM {$wpdb->prefix}bp_follow WHERE date_recorded >= %s",
                $date_range
            ));
            
            echo '<div class="bb-stats-grid">';
            echo '<div class="bb-stat-card">';
            echo '<div class="bb-stat-value">' . number_format($total_follows) . '</div>';
            echo '<div class="bb-stat-label">Total Follows</div>';
            echo '</div>';
            
            echo '<div class="bb-stat-card">';
            echo '<div class="bb-stat-value">' . number_format($new_follows) . '</div>';
            echo '<div class="bb-stat-label">New Follows</div>';
            echo '</div>';
            echo '</div>';
            
            echo '</div>';
        }
        
        if (!$this->table_exists($wpdb->prefix . 'bp_friends') && !$this->table_exists($wpdb->prefix . 'bp_follow')) {
            echo '<div class="bb-notice bb-notice-warning">';
            echo '<p>No network/connection data available. Friends and Follow components may not be active.</p>';
            echo '</div>';
        }
        
        echo '</div>';
    }

    private function display_engagement_section() {
        global $wpdb;
        
        echo '<div class="bb-analytics-section">';
        echo '<h2><i class="dashicons dashicons-heart"></i> Engagement Analytics</h2>';
        
        $date_range = $this->get_date_range_sql();
        
        // User sessions (if custom table exists)
        if ($this->table_exists($wpdb->prefix . 'bb_user_sessions')) {
            echo '<div class="bb-subsection">';
            echo '<h3>User Sessions</h3>';
            
            $avg_session_duration = $wpdb->get_var($wpdb->prepare(
                "SELECT AVG(duration) FROM {$wpdb->prefix}bb_user_sessions WHERE session_start >= %s",
                $date_range
            ));
            
            $avg_posts_viewed = $wpdb->get_var($wpdb->prepare(
                "SELECT AVG(posts_viewed) FROM {$wpdb->prefix}bb_user_sessions WHERE session_start >= %s",
                $date_range
            ));
            
            echo '<div class="bb-stats-grid">';
            echo '<div class="bb-stat-card">';
            echo '<div class="bb-stat-value">' . round($avg_session_duration / 60, 1) . ' min</div>';
            echo '<div class="bb-stat-label">Avg Session Duration</div>';
            echo '</div>';
            
            echo '<div class="bb-stat-card">';
            echo '<div class="bb-stat-value">' . round($avg_posts_viewed, 1) . '</div>';
            echo '<div class="bb-stat-label">Avg Posts per Session</div>';
            echo '</div>';
            echo '</div>';
            
            echo '</div>';
        }
        
        // Activity engagement
        if ($this->table_exists($wpdb->prefix . 'bp_activity')) {
            echo '<div class="bb-subsection">';
            echo '<h3>Activity Engagement</h3>';
            
            $total_activities = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM {$wpdb->prefix}bp_activity WHERE date_recorded >= %s",
                $date_range
            ));
            
            $unique_contributors = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(DISTINCT user_id) FROM {$wpdb->prefix}bp_activity WHERE date_recorded >= %s",
                $date_range
            ));
            
            $avg_activities_per_user = $unique_contributors > 0 ? $total_activities / $unique_contributors : 0;
            
            echo '<div class="bb-stats-grid">';
            echo '<div class="bb-stat-card">';
            echo '<div class="bb-stat-value">' . number_format($total_activities) . '</div>';
            echo '<div class="bb-stat-label">Total Activities</div>';
            echo '</div>';
            
            echo '<div class="bb-stat-card">';
            echo '<div class="bb-stat-value">' . number_format($unique_contributors) . '</div>';
            echo '<div class="bb-stat-label">Active Contributors</div>';
            echo '</div>';
            
            echo '<div class="bb-stat-card">';
            echo '<div class="bb-stat-value">' . round($avg_activities_per_user, 1) . '</div>';
            echo '<div class="bb-stat-label">Avg Activities per User</div>';
            echo '</div>';
            echo '</div>';
            
            echo '</div>';
        }
        
        echo '</div>';
    }

    private function display_search_section() {
        global $wpdb;
        
        echo '<div class="bb-analytics-section">';
        echo '<h2><i class="dashicons dashicons-search"></i> Search Analytics</h2>';
        
        if ($this->table_exists($wpdb->prefix . 'bb_search_log')) {
            $date_range = $this->get_date_range_sql();
            
            // Search overview
            echo '<div class="bb-subsection">';
            echo '<h3>Search Overview</h3>';
            
            $total_searches = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM {$wpdb->prefix}bb_search_log WHERE search_date >= %s",
                $date_range
            ));
            
            $unique_searchers = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(DISTINCT user_id) FROM {$wpdb->prefix}bb_search_log WHERE search_date >= %s AND user_id IS NOT NULL",
                $date_range
            ));
            
            echo '<div class="bb-stats-grid">';
            echo '<div class="bb-stat-card">';
            echo '<div class="bb-stat-value">' . number_format($total_searches) . '</div>';
            echo '<div class="bb-stat-label">Total Searches</div>';
            echo '</div>';
            
            echo '<div class="bb-stat-card">';
            echo '<div class="bb-stat-value">' . number_format($unique_searchers) . '</div>';
            echo '<div class="bb-stat-label">Unique Searchers</div>';
            echo '</div>';
            echo '</div>';
            
            echo '</div>';
            
            // Top search terms
            echo '<div class="bb-subsection">';
            echo '<h3>Top Search Terms</h3>';
            
            $top_searches = $wpdb->get_results($wpdb->prepare("
                SELECT search_term, COUNT(*) as search_count
                FROM {$wpdb->prefix}bb_search_log
                WHERE search_date >= %s
                GROUP BY search_term
                ORDER BY search_count DESC
                LIMIT 20
            ", $date_range));
            
            if (!empty($top_searches)) {
                echo '<div class="bb-table-responsive">';
                echo '<table class="bb-analytics-table">';
                echo '<thead><tr><th>Search Term</th><th>Count</th></tr></thead>';
                echo '<tbody>';
                
                foreach ($top_searches as $search) {
                    echo '<tr>';
                    echo '<td>' . esc_html($search->search_term) . '</td>';
                    echo '<td>' . number_format($search->search_count) . '</td>';
                    echo '</tr>';
                }
                echo '</tbody></table>';
                echo '</div>';
            }
            
            echo '</div>';
        } else {
            echo '<div class="bb-notice bb-notice-info">';
            echo '<p>Search logging is not enabled. To track search analytics, the search logging feature needs to be activated.</p>';
            echo '</div>';
        }
        
        echo '</div>';
    }

    private function display_messages_section() {
        global $wpdb;
        
        echo '<div class="bb-analytics-section">';
        echo '<h2><i class="dashicons dashicons-email"></i> Messages Analytics</h2>';
        
        if ($this->table_exists($wpdb->prefix . 'bp_messages_messages')) {
            $date_range = $this->get_date_range_sql();
            
            // Messages overview
            echo '<div class="bb-subsection">';
            echo '<h3>Messages Overview</h3>';
            
            $total_messages = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM {$wpdb->prefix}bp_messages_messages WHERE date_sent >= %s",
                $date_range
            ));
            
            $unique_senders = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(DISTINCT sender_id) FROM {$wpdb->prefix}bp_messages_messages WHERE date_sent >= %s",
                $date_range
            ));
            
            $total_threads = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(DISTINCT thread_id) FROM {$wpdb->prefix}bp_messages_messages WHERE date_sent >= %s",
                $date_range
            ));
            
            echo '<div class="bb-stats-grid">';
            echo '<div class="bb-stat-card">';
            echo '<div class="bb-stat-value">' . number_format($total_messages) . '</div>';
            echo '<div class="bb-stat-label">Total Messages</div>';
            echo '</div>';
            
            echo '<div class="bb-stat-card">';
            echo '<div class="bb-stat-value">' . number_format($unique_senders) . '</div>';
            echo '<div class="bb-stat-label">Active Senders</div>';
            echo '</div>';
            
            echo '<div class="bb-stat-card">';
            echo '<div class="bb-stat-value">' . number_format($total_threads) . '</div>';
            echo '<div class="bb-stat-label">Conversations</div>';
            echo '</div>';
            echo '</div>';
            
            echo '</div>';
            
            // Top message senders
            echo '<div class="bb-subsection">';
            echo '<h3>Most Active Messagers</h3>';
            
            $top_senders = $wpdb->get_results($wpdb->prepare("
                SELECT 
                    u.display_name,
                    COUNT(m.id) as message_count
                FROM {$wpdb->prefix}bp_messages_messages m
                JOIN {$wpdb->users} u ON m.sender_id = u.ID
                WHERE m.date_sent >= %s
                GROUP BY m.sender_id, u.display_name
                ORDER BY message_count DESC
                LIMIT 10
            ", $date_range));
            
            if (!empty($top_senders)) {
                echo '<div class="bb-table-responsive">';
                echo '<table class="bb-analytics-table">';
                echo '<thead><tr><th>User</th><th>Messages Sent</th></tr></thead>';
                echo '<tbody>';
                
                foreach ($top_senders as $sender) {
                    echo '<tr>';
                    echo '<td>' . esc_html($sender->display_name) . '</td>';
                    echo '<td>' . number_format($sender->message_count) . '</td>';
                    echo '</tr>';
                }
                echo '</tbody></table>';
                echo '</div>';
            }
            
            echo '</div>';
        } else {
            echo '<div class="bb-notice bb-notice-warning">';
            echo '<p>BuddyPress/BuddyBoss messages component is not active or tables are missing.</p>';
            echo '</div>';
        }
        
        echo '</div>';
    }

    /**
     * Get all chart data for JavaScript
     */
    private function get_all_chart_data() {
        return array(
            'membersGrowth' => $this->get_members_growth_data(),
            'memberTypes' => $this->get_member_types_chart_data(),
            'activityOverTime' => $this->get_activity_over_time_data(),
            'activityTypes' => $this->get_activity_types_chart_data()
        );
    }

    private function get_members_growth_data() {
        global $wpdb;
        
        $cache_key = 'bb_analytics_members_growth';
        return $this->get_cached_data($cache_key, function() use ($wpdb) {
            $data = $wpdb->get_results("
                SELECT 
                    DATE_FORMAT(user_registered, '%Y-%m') as month, 
                    COUNT(*) as new_members,
                    (SELECT COUNT(*) FROM {$wpdb->users} u2 WHERE u2.user_registered <= LAST_DAY(STR_TO_DATE(CONCAT(DATE_FORMAT(u1.user_registered, '%Y-%m'), '-01'), '%Y-%m-%d'))) as total_members
                FROM {$wpdb->users} u1
                WHERE user_registered >= DATE_SUB(NOW(), INTERVAL 12 MONTH)
                GROUP BY DATE_FORMAT(user_registered, '%Y-%m')
                ORDER BY month
            ");
            
            return $data;
        });
    }

    private function get_member_types_chart_data() {
        global $wpdb;
        
        $cache_key = 'bb_analytics_member_types';
        return $this->get_cached_data($cache_key, function() use ($wpdb) {
            $data = $wpdb->get_results("
                SELECT 
                    COALESCE(um.meta_value, 'Subscriber') as member_type,
                    COUNT(*) as count
                FROM {$wpdb->users} u
                LEFT JOIN {$wpdb->usermeta} um ON u.ID = um.user_id AND um.meta_key = 'member_type'
                GROUP BY COALESCE(um.meta_value, 'Subscriber')
                ORDER BY count DESC
            ");
            
            return $data;
        });
    }

    private function get_activity_over_time_data() {
        global $wpdb;
        
        if (!$this->table_exists($wpdb->prefix . 'bp_activity')) {
            return array();
        }
        
        $cache_key = 'bb_analytics_activity_time';
        return $this->get_cached_data($cache_key, function() use ($wpdb) {
            $data = $wpdb->get_results("
                SELECT 
                    DATE_FORMAT(date_recorded, '%Y-%m-%d') as date, 
                    COUNT(*) as count
                FROM {$wpdb->prefix}bp_activity
                WHERE date_recorded >= DATE_SUB(NOW(), INTERVAL 30 DAY)
                GROUP BY DATE_FORMAT(date_recorded, '%Y-%m-%d')
                ORDER BY date
            ");
            
            return $data;
        });
    }

    private function get_activity_types_chart_data() {
        global $wpdb;
        
        if (!$this->table_exists($wpdb->prefix . 'bp_activity')) {
            return array();
        }
        
        $cache_key = 'bb_analytics_activity_types';
        return $this->get_cached_data($cache_key, function() use ($wpdb) {
            $date_range = $this->get_date_range_sql();
            $data = $wpdb->get_results($wpdb->prepare("
                SELECT type, COUNT(*) as count
                FROM {$wpdb->prefix}bp_activity
                WHERE date_recorded >= %s
                GROUP BY type
                ORDER BY count DESC
            ", $date_range));
            
            return $data;
        });
    }

    /**
     * Helper functions
     */
    private function table_exists($table_name) {
        global $wpdb;
        $table = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_name));
        return $table === $table_name;
    }

    private function format_activity_type($type) {
        $formatted_types = array(
            'activity_update' => 'Status Updates',
            'activity_comment' => 'Comments',
            'friendship_accepted' => 'Friendships',
            'friendship_created' => 'Friend Requests',
            'created_group' => 'Group Created',
            'joined_group' => 'Group Joined',
            'group_details_updated' => 'Group Updated',
            'new_blog_post' => 'Blog Posts',
            'new_blog_comment' => 'Blog Comments',
            'new_forum_topic' => 'Forum Topics',
            'new_forum_post' => 'Forum Posts'
        );
        
        return isset($formatted_types[$type]) ? $formatted_types[$type] : ucwords(str_replace('_', ' ', $type));
    }
}

// Initialize the plugin
new BuddyBoss_Complete_Analytics_Improved();

